import os, sys
from gpaw import GPAW, PW, FermiDirac
from ase import Atoms
from ase.optimize import QuasiNewton, BFGS
from ase.constraints import FixAtoms
import numpy as np
from scipy.optimize import curve_fit
from ase.parallel import world
sys.path.append(os.getcwd())
from geomaker import *

# define function that runs a DFT geometry optimization.  Returns the energy
def get_opt_energy(slab, k=3):
    """ sets up and runs a DFT geometry optimization and returns the final energy.
    Input:
    - slab: ase geometry object of the system to be optimized and to compute the energy of
    - k: number of k-points in x and y direction
    Output:
    - energy [eV]
    """
    # set up calculation in GPAW
    if len(slab)==2:
        outname = "H2"
    else:
        outname = "Pt111_{}".format(sum(slab.get_tags()==0)) #labels the output with the number of Hads
    if world.rank == 0:
        print(outname)
    calc = GPAW(#setups = {'Pt':'10'}, # use pseudo-potential with only 10 valence e- for Pt
                txt = outname + ".out", 
                mode = PW(340),
                xc='PBE', # define functional
                kpts=(k, k, 1), #define number of k-points
                occupations={'name': 'fermi-dirac', 'width': 0.03},
                #symmetry = {'point_group': False} #turn off point group symmetry to avoid problems with symmetry breaking during opt
            )
    # add the calculator to your atoms object
    slab.calc = calc

    # setup the geometry relaxation
    #relax = QuasiNewton(slab, trajectory = outname + ".traj")
    relax = BFGS(slab, trajectory = outname + ".traj")
    force_convergence_threshold = 0.07 #eV/A  --- note that this is rather large for such a simple system
    relax.run(fmax=force_convergence_threshold)

    # extract final energy
    return slab.get_potential_energy()

if __name__  == "__main__":
    # compute total energy and binding energy for all surfaces (systems)
    # Define surface unit cell and max number of adsorbates
    Nx = 4
    Ny = 4
    l = 4
    Nmax = Nx * Ny

    #perform slab calculations
    energies = np.zeros(Nmax + 1)
    for N in range(Nmax + 1):
        # build slab and compute energy otherwise
        slab = build_slab_with_adsorbate(N, Nx, Ny, l)
        energies[N] = get_opt_energy(slab, k=10)
    # save result to file
    if world.rank == 0:
        np.save('slab_energies.npy', energies)

    #build molecule and compute energy
    h2 = build_h2()
    h2_energy = get_opt_energy(h2, k=1)
    # save result to file
    if world.rank == 0:
        np.save('h2_energy.npy', h2_energy)

    # print total energies
    if world.rank == 0:
        print("Total energies")
        print("slabs: ", energies)
        print("H2:    ",h2_energy)




