import os, sys
from gpaw import GPAW, PW, FermiDirac
from ase import Atoms
from ase.optimize import QuasiNewton, BFGS
from ase.constraints import FixAtoms
import numpy as np
from scipy.optimize import curve_fit
from ase.parallel import world
sys.path.append(os.getcwd())
from geomaker import *

# define function that runs a DFT geometry optimization.  Returns the energy
def get_opt_energy(slab, k=3):
    """ sets up and runs a DFT geometry optimization and returns the final energy.
    Input:
    - slab: ase geometry object of the system to be optimized and to compute the energy of
    - k: number of k-points in x and y direction
    Output:
    - energy [eV]
    """
    # set up calculation in GPAW
    if len(slab)==2:
        outname = "H2"
    else:
        outname = "Pt111_{}".format(sum(slab.get_tags()==0)) #labels the output with the number of Hads
    if world.rank == 0:
        print(outname)
    calc = GPAW(#setups = {'Pt':'10'}, # use pseudo-potential with only 10 valence e- for Pt
                txt = outname + ".out", 
                mode = PW(340),
                #mode='fd', # use finite difference mode
                #h = 0.22, # grid spacing
                xc='PBE', # define functional
                kpts=(k, k, 1), #define number of k-points
                #symmetry = {'point_group': False} #turn off point group symmetry to avoid problems with symmetry breaking during opt
            )
    # add the calculator to your atoms object
    slab.calc = calc

    # setup the geometry relaxation
    #relax = QuasiNewton(slab, trajectory = outname + ".traj")
    relax = BFGS(slab, trajectory = outname + ".traj")
    force_convergence_threshold = 0.07 #eV/A  --- note that this is rather large for such a simple system
    relax.run(fmax=force_convergence_threshold)

    # extract final energy
    return slab.get_potential_energy()

if __name__  == "__main__":
    # compute total energy and binding energy for all surfaces (systems)
    # Define surface unit cell and max number of adsorbates
    Nx = 2
    Ny = 2
    l = 4
    Nmax = Nx * Ny

    # check whether some energies have already been computed and written to file
    if os.path.exists('slab_energies.npy'):
        energies = np.load('slab_energies.npy')
    else:
        energies = np.zeros(Nmax + 1)

    #perform slab calculations
    for N in range(Nmax + 1):
        # skip calculations that have already been done
        if abs(energies[N])> 1E-5: 
            continue 
        # build slab and compute energy otherwise
        slab = build_slab_with_adsorbate(N, Nx, Ny, l)
        energies[N] = get_opt_energy(slab, k=4)
    # save result to file
    if world.rank == 0:
        np.save('slab_energies.npy', energies)

    # check wheter H2 has already been computed
    if os.path.exists('h2_energy.npy'):
        h2_energy = np.load('h2_energy')
    else:
        #build molecule and compute energy
        h2 = build_H2()
        h2_energy = get_opt_energy(h2, k=1)
        # save result to file
        if world.rank == 0:
            np.save('h2_energy.npy', h2_energy)

    # print total energies
    if world.rank == 0:
        print("Total energies")
        print("slabs: ", energies)
        print("H2:    ",h2_energy)

    # compute binding energies
    binding_eneries = np.zeros(Nmax)
    for N in range(1, Nmax+1):
        binding_energies[N-1] = energies[N] - energies[0] - float(N)/2*energy_h2

    #print binding energies
    if world.rank == 0:
        print("Binding energies")
        print("All:  ", binding_energies)
        np.save('binding_energies.npy', binding_energies)



